package Ex5_7
function EXP_1
  "Return the exponential of a matrix by adaptive Taylor series expansion with scaling and balancing"

  extends Modelica.Icons.Function;
  extends Modelica.Math.Matrices;/*NEW-Without this the balance command will not work*/
  input Real A[:, size(A, 1)];
  input Real T=1;
  output Real phi[size(A, 1), size(A, 1)] "= exp(A*T)";

protected
  parameter Integer nmax=21;
  /*max number of iterations*/
  Integer na=size(A, 1);/*Removing the parameter command from the main(exp)file*/
  Integer j=1;
  Integer k=0;
  Boolean done=false;
  Real Anorm;
  Real Tscaled=1;
  Real Atransf[na, na];
  Real D[na, na];
  /*D: dummy variable for psi*/
  Real M[na, na];
  /*M: dummy matrix*/
  Real Diag[na];
  /*diagonal transformation matrix for balancing*/

  encapsulated function columnNorm "Returns the column norm of a matrix"
    import Modelica;
    extends Modelica.Icons.Function;
    input Real A[:, :] "Input matrix";
    output Real result=0.0 "1-norm of matrix A";
  algorithm
    for i in 1:size(A, 2) loop
      result := max(result, sum(abs(A[:, i])));
    end for;
  end columnNorm;

algorithm
  // balancing of A
  (Diag,Atransf) := balance(A);

  // scaling of T until norm(A)*/(2^k) < 1
  Tscaled := T;
  /*Anorm: column-norm of matrix A*/
  Anorm := columnNorm(Atransf);
  Anorm := Anorm*T;
  while Anorm >= 0.5 loop
    Anorm := Anorm/2;
    Tscaled := Tscaled/2;
    k := k + 1;
  end while;

  // Computation of psi by Taylor-series approximation
  M := identity(na);
  D := M;
  while j < nmax and not done loop
    M := Atransf*M*Tscaled/j;
    //stop if the new element of the series is small
    if columnNorm((D + M) - D) == 0 then
      done := true;
    else
      D := M + D;
      j := j + 1;
    end if;
  end while;

  // re-scaling
  for i in 1:k loop
    D := D*D;
  end for;

  // re-balancing: psi := diagonal(Diag)*D*inv(diagonal(Diag));
  for j in 1:na loop
    for k in 1:na loop
      phi[j, k] := D[j, k]*Diag[j]/Diag[k];
    end for;
  end for;
  annotation (Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
phi = Matrices.<strong>exp</strong>(A);
phi = Matrices.<strong>exp</strong>(A,T=1);
</pre></blockquote>

<h4>Description</h4>
<p>
This function computes the exponential e<sup><strong>A</strong>T</sup> of matrix <strong>A</strong>, i.e.
</p>
<blockquote>
<pre>                            (<strong>A</strong>T)^2   (<strong>A</strong>T)^3
   <font size=4> <strong>&Phi;</strong></font> = e^(<strong>A</strong>T) = <strong>I</strong> + <strong>A</strong>T + ------ + ------ + ....
                            2!       3!
</pre></blockquote>

<p>where e=2.71828..., <strong>A</strong> is an n x n matrix with real elements and T is a real number,
e.g., the sampling time.
<strong>A</strong> may be singular. With the exponential of a matrix it is, e.g., possible
to compute the solution of a linear system of differential equations</p>
<pre>    der(<strong>x</strong>) = <strong>A</strong>*<strong>x</strong>   ->   <strong>x</strong>(t0 + T) = e^(<strong>A</strong>T)*x(t0)
</pre>

<h4>Algorithmic details:</h4>

<p>The algorithm is taken from</p>
<dl>
<dt>H. D. Joos, G. Gr&uuml;bel:</dt>
<dd><strong>RASP'91 Regulator Analysis and Synthesis Programs</strong><br>
  DLR - Control Systems Group 1991</dd>
</dl>
<p>The following steps are performed to calculate the exponential of A:</p>
<ol>
<li>Matrix <strong>A</strong> is balanced<br>
(= is transformed with a diagonal matrix <strong>D</strong>, such that inv(<strong>D</strong>)*<strong>A</strong>*<strong>D</strong>
has a smaller condition as <strong>A</strong>).</li>
<li>The scalar T is divided by a multiple of 2 such that norm(
     inv(<strong>D</strong>)*<strong>A</strong>*<strong>D</strong>*T/2^k ) &lt; 0.5. Note, that (1) and (2) are implemented such that no round-off errors
are introduced.</li>
<li>The matrix from (2) is approximated by explicitly performing the Taylor
series expansion with a variable number of terms.
Truncation occurs if a new term does no longer contribute to the value of <strong>&Phi;</strong>
from the previous iteration.</li>
<li>The resulting matrix is transformed back, by reverting the steps of (2)
and (1).</li>
</ol>
<p>In several sources it is not recommended to use Taylor series expansion to
calculate the exponential of a matrix, such as in 'C.B. Moler and C.F. Van Loan:
Nineteen dubious ways to compute the exponential of a matrix. SIAM Review 20,
pp. 801-836, 1979' or in the documentation of m-file expm2 in MATLAB version 6
(<a href=\"http://www.mathworks.com\">http://www.mathworks.com</a>) where it is
stated that 'As a practical numerical method, this is often slow and inaccurate'.
These statements are valid for a direct implementation of the Taylor series
expansion, but <em>not</em> for the implementation variant used in this function.
</p>

</html>", revisions="<html>
<p><strong>Release Notes:</strong></p>
<ul>
<li><em>July 5, 2002</em>
     by H. D. Joos and Nico Walther<br>
     Implemented.
</li>
</ul>
</html>"));
end EXP_1;

class Ex5_7
/*function EXP_1 has been used to solve this problem*/
/*solution from differential equation*/
Real x[2];/*state vector*/
Real A[2,2]=[-2,1;1,-2];/*state matrix*/
Real B[2]={1,1};/*input matrix*/
Real u=0/*zero input response*/;

/*analytical x*/
Real x_ana[2];/*state vector for analytical solution*/
Real x_0[2]={10,20};/*assuming initial value of x*/

initial equation
x={10,20};/*assuming*/
equation
der(x)=A*x+B*u;

/*analytical solution of the above differential equation*/
x_ana=(EXP_1(A*time))*x_0;

end Ex5_7;
end Ex5_7;
